﻿#include "precompiled.h"
#include "helpers.h"

#include "ConsoleDefines.h"

namespace RTCam {

string StringFormat(_In_z_ const char* format, ...) {
	va_list args;
	va_start(args, format);

	// Add 1 to length because _vscprintf doesn't count terminating '\0'
	size_t bufferLength = _vscprintf(format, args) + 1;

	unique_ptr<char[]> buffer(new char[bufferLength]);
	vsprintf_s(buffer.get(), bufferLength, format, args);

	va_end(args);

	// Make sure the string is terminated properly to prevent buffer overflow.
	ASSERT(buffer[bufferLength - 1] == '\0');

	return string(buffer.get());
}

wstring WstringFormat(_In_z_ const wchar_t* format, ... )
{
	va_list args;
	va_start(args, format);

	// Add 1 to length because _vscwprintf doesn't count terminating '\0'
	size_t bufferLength = _vscwprintf(format, args) + 1;

	unique_ptr<wchar_t[]> buffer(new wchar_t[bufferLength]);
	vswprintf_s(buffer.get(), bufferLength, format, args);

	va_end(args);

	// Make sure the string is terminated properly to prevent buffer overflow.
	ASSERT(buffer[bufferLength - 1] == '\0');

	return wstring(buffer.get());
}


void DPrintf(_In_z_ const char* format, ...)
{
	va_list args;
	va_start(args, format);

	// Add 1 to length because _vscprintf doesn't count terminating '\0'
	size_t bufferLength = _vscprintf(format, args) + 1;

	unique_ptr<char[]> buffer(new char[bufferLength]);
	vsprintf_s(buffer.get(), bufferLength, format, args);

	va_end(args);

	// Make sure the string is terminated properly to prevent buffer overflow.
	ASSERT(buffer[bufferLength - 1] == '\0');

	wstring messageW = Utf8ToUtf16(buffer.get());
	OutputDebugString(messageW.c_str());
#ifdef SHOW_CONSOLE
	printf_s(buffer.get());
#endif
}

void DPrintf(_In_z_ const wchar_t* format, ...)
{
	va_list args;
	va_start(args, format);

	// Add 1 to length because _vscwprintf doesn't count terminating '\0'
	size_t bufferLength = _vscwprintf(format, args) + 1;

	unique_ptr<wchar_t[]> buffer(new wchar_t[bufferLength]);
	vswprintf_s(buffer.get(), bufferLength, format, args);

	va_end(args);

	// Make sure the string is terminated properly to prevent buffer overflow.
	ASSERT(buffer[bufferLength - 1] == '\0');

	OutputDebugString(buffer.get());
#ifdef SHOW_CONSOLE
	wprintf_s(buffer.get());
#endif
}

wstring GetErrorMessage(HRESULT hr)
{
	LPVOID pBuffer;

	int ret = FormatMessage(  
		FORMAT_MESSAGE_ALLOCATE_BUFFER | // The function will allocate space for pBuffer.
		FORMAT_MESSAGE_FROM_SYSTEM, // System wide message.
		NULL, // Message is not in a module.
		hr, // Message identifier.
		MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT), // Default language.
		(LPWSTR)&pBuffer, // Buffer to hold the text string.
		0, // The function will allocate at least this much for pBuffer.
		NULL // No inserts.
		);

	wstring errorMsg;

	if(ret) {
		errorMsg = wstring((LPWSTR)pBuffer) + L"\n";
	} else {
		errorMsg = L"Error: 0x" + std::to_wstring(hr) + L"\n";
	}
	LocalFree(pBuffer);

	return errorMsg;
}

wstring GetCurPath()
{
	wchar_t buffer[MAX_PATH];
	GetCurrentDirectory(MAX_PATH, buffer);
	wstring path(buffer);
	wstring::size_type pos = path.find_last_of(L"\\/") + 1;
	return path.substr(0, pos);
}

wstring GetExePath()
{
	wchar_t buffer[MAX_PATH];
	GetModuleFileName(nullptr, buffer, MAX_PATH);
	wstring path(buffer);
	wstring::size_type pos = path.find_last_of(L"\\/") + 1;
	return path.substr(0, pos);
}

bool IsWhitespace(string str)
{
	return std::find_if(str.begin(), str.end(), [](char c) { return !isspace(c); } ) == str.end();
}

} // end namespace